#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include "hopcroft.h"
#include "cdaig.h"





char* get_Istring(int pos, bool is_first, int n_Icharacter, int length, FILE *data, int n, int m);
char* get_Icharacter(int pos, bool is_first, int length, FILE* data, int n, int m);
int32_t istrcmp(char *seq1, char *seq2);
int32_t find_edge(struct node* cdaig, uint32_t q, char* isuffix, FILE* data, int n, int m);
struct edge_partition slow_find(uint32_t p, FILE* data, uint32_t e, struct node* cdaig, int n, int m);
void add_edge(struct node* cdaig, struct edge_partition e, uint32_t dest);
bool same_partition_array(struct partition* A, size_t len_A, struct partition* B, size_t len_B);
bool same_set(struct partition* a, struct partition* b);
int compare_uint32(const void* a, const void* b);
void set_union(struct partition* set_of_sets, uint32_t n_set_of_sets, struct partition* new_set, uint32_t n_new_set, struct partition* final_set, uint32_t n_final_set, int32_t index);
struct partition_vect* split(struct node* cdaig, struct partition s, struct partition* p, uint32_t np, FILE* data, int n, int m);
void hopcroft(struct node** cdaig, uint32_t* v, FILE* data, int n, int m);
void print_cdaig(struct node* cdaig, uint32_t v);
uint32_t cdaig_generation(struct node** cdaig, FILE* data, int n, int m, int* suff);



/**
 * Function to export the CDAIG structure in DOT format.
 * 
 * Parameters:
 *     · cdaig: Pointer to the array of nodes representing the DFA.
 *     · num_nodes: Number of nodes in the DFA.
 *     · filename: Name of the output DOT file.
 *     · data: File pointer to the input data file.
 *     · n: Number of rows in the original data matrix.
 *     · m: Number of columns in the original data matrix.
 * 
 * Notes:
 *     - If an edge starts at position n*m, its label will be '$'.
 *     - If an edge points to node 0, a '$' is appended to its label.
 */
void generate_dot(struct node* cdaig, uint32_t num_nodes, const char* filename, FILE* data, int n, int m) {
    uint32_t i, j;
    FILE* dot_file = NULL;
    uint32_t total_size = n * m;

    dot_file = fopen(filename, "w");
    if (dot_file == NULL) {
        printf("Error opening .dot file.\n");
        return;
    }

    fprintf(dot_file, "digraph CDAIG {\n");

    for (i = 0; i < num_nodes; i++) {
        fprintf(dot_file, "    %d [label=\"Node %d\"];\n", i, i);

        for (j = 0; j < cdaig[i].n_edges; j++) {
            struct edge e = cdaig[i].edges[j];
            char* label = NULL;

            if (e.beginpos == total_size) {
                /* Special case: edge represents only the '$' symbol */
                label = strdup("$");
            } else if (e.beginpos > total_size) {
                printf("Error: Invalid beginpos at node %d, edge %d\n", i, j);
                continue;
            } else {
                /* Get the istring corresponding to the edge label */
                label = get_Istring(e.beginpos, e.is_first, e.ilength, e.length, data, n, m);

                /* Append '$' if destination is node 0 */
                if (e.dest == 0) {
                    size_t len = strlen(label);
                    char* label_with_dollar = (char*)malloc(len + 2); /* +1 for '$' and +1 for '\0' */
                    strcpy(label_with_dollar, label);
                    label_with_dollar[len] = '$';
                    label_with_dollar[len + 1] = '\0';
                    free(label);
                    label = label_with_dollar;
                }
            }

            fprintf(dot_file, "    %d -> %d [label=\"%s\"];\n", i, e.dest, label);
            free(label);
        }
    }

    fprintf(dot_file, "}\n");
    fclose(dot_file);
}

/**
 * Function that prints the information stored on the cdaig
 *  Param:
 *      cdaig: the adjacency list containing the information of the cdaig.
 *      v: Number of vertices of the cdaig
 * 
 *  The function does not return anything
 */
void print_cdaig(struct node* cdaig, uint32_t v) {
    for (uint32_t i = 0; i < v; i++) {
        printf("Node %u (s = %u):\n", i, cdaig[i].s);
        for (uint32_t j = 0; j < cdaig[i].n_edges; j++) {
            struct edge e = cdaig[i].edges[j];
            printf("  -> Edge to %u: beginpos=%u, length=%u, ilength=%u, is_first=%s\n",
                e.dest, e.beginpos, e.length, e.ilength, e.is_first ? "true" : "false");
        }
    }
}
/***********************************************************************************
 ***********************************************************************************/

int main(int argc, char *argv[]) {
    
    if(argc != 4){
        printf("Usage: %s <file> <n_rows> <n_columns>\n", argv[0]);
        return 1;
    }

    FILE* data = NULL;
    struct node* cdaig;
    uint32_t i, e = 0, n = (uint32_t) strtoul(argv[2], NULL, 10), m = (uint32_t) strtoul(argv[3], NULL, 10);
    int len;
    int suff = 0;
    char *dot_filename;
     


    len = strlen(argv[1]) + 11;
    dot_filename = malloc(len);
    if(!dot_filename){
        printf("Error allocating memory for the name file\n");
        exit(1);
    }

    strcpy(dot_filename, argv[1]);
    strcat(dot_filename, ".cdaig.dot");
    



    data = fopen(argv[1], "rb");
    if (!data) {
        printf("Error opening data file.\n");
        return 1;
    }
    printf("File ready to be used\n");
    cdaig = NULL;

    
    uint32_t num_nodes = cdaig_generation(&cdaig, data, n, m, &suff);

    
    for (i = 0; i < num_nodes; i++) {
        e += cdaig[i].n_edges;
    }
    printf("v=%d e=%d suff=%d\n", num_nodes, e, suff);

    

    /*print_cdaig(cdaig, num_nodes);*/  
    generate_dot(cdaig, num_nodes, dot_filename, data, n, m);

    printf("File generated: %s\n", dot_filename);
    

    
    for (i = 0; i < num_nodes; i++) {
        free(cdaig[i].edges);
    }
    free(cdaig);
    fclose(data);

    return 0;
}
